"use client";
import { Check, ChevronsUpDown, Loader2 } from "lucide-react";
import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import { useEffect, useState } from "react";
import { useInView } from "react-intersection-observer";
import { t } from "@/utils";
import { BiPlanet } from "react-icons/bi";
import { FaSearch } from "react-icons/fa";
import { useRouter } from "next/navigation";
import useHomeCategories from "@/components/Layout/useHomeCategories";
import { getCurrentLangCode } from "@/redux/reducer/languageSlice";
import { useSelector } from "react-redux";

const Search = () => {
  const { cateData, isLoadMore, lastPage, currentPage, getCategories } =
    useHomeCategories({ skipInitialLoad: false });
  const router = useRouter();
  const categoryList = [
    { slug: "all-categories", translated_name: "All Categories" },
    ...cateData,
  ];
  const [open, setOpen] = useState(false);
  const [value, setValue] = useState("all-categories");
  const selectedItem = categoryList.find((item) => item.slug === value);
  const hasMore = currentPage < lastPage;
  const { ref, inView } = useInView();
  const [searchQuery, setSearchQuery] = useState("");
  const langCode = useSelector(getCurrentLangCode);

  useEffect(() => {
    if (open && inView && hasMore && !isLoadMore) {
      getCategories(currentPage + 1);
    }
  }, [hasMore, inView, isLoadMore, open]);

  const handleSearchNav = (e) => {
    e.preventDefault();

    const query = encodeURIComponent(searchQuery);

    if (selectedItem?.slug === "all-categories") {
      router.push(`/ads?query=${query}&lang=${langCode}`);
    } else {
      router.push(
        `/ads?category=${selectedItem?.slug}&query=${query}&lang=${langCode}}`
      );
    }
  };

  return (
    <>
      <Popover open={open} onOpenChange={setOpen}>
        <PopoverTrigger asChild>
          <Button
            variant="outline"
            role="combobox"
            aria-expanded={open}
            className="min-w-[125px] max-w-[125px] sm:min-w-[156px] sm:max-w-[156px] py-1 px-1.5 sm:py-2 sm:px-3 justify-between border-none hover:bg-transparent font-normal"
          >
            <span className="truncate">
              {selectedItem?.translated_name || t("selectCat")}
            </span>
            <ChevronsUpDown className="opacity-50" />
          </Button>
        </PopoverTrigger>
        <PopoverContent align="start" className="w-[200px] p-0">
          <Command>
            <CommandInput placeholder={t("searchACategory")} />
            <CommandList>
              <CommandEmpty>{t("noCategoryFound")}</CommandEmpty>
              <CommandGroup>
                {categoryList.map((category, index) => {
                  const isLast = open && index === categoryList.length - 1;
                  return (
                    <CommandItem
                      key={category?.slug}
                      value={category?.slug}
                      onSelect={(currentValue) => {
                        setValue(currentValue);
                        setOpen(false);
                      }}
                      ref={isLast ? ref : null}
                    >
                      {category.translated_name || category?.name}
                      <Check
                        className={cn(
                          "ml-auto",
                          value === category.slug ? "opacity-100" : "opacity-0"
                        )}
                      />
                    </CommandItem>
                  );
                })}
              </CommandGroup>
              {isLoadMore && (
                <div className="flex justify-center items-center pb-2 text-muted-foreground">
                  <Loader2 className="animate-spin" />
                </div>
              )}
            </CommandList>
          </Command>
        </PopoverContent>
      </Popover>
      <form
        onSubmit={handleSearchNav}
        className="w-full flex items-center gap-2 ltr:border-l rtl:border-r py-1 px-1.5 sm:py-2 sm:px-3"
      >
        <BiPlanet color="#595B6C" className="min-w-4 min-h-4" />
        <input
          type="text"
          placeholder={t("searchAd")}
          className="text-sm outline-none w-full"
          value={searchQuery}
          onChange={(e) => setSearchQuery(e.target.value)}
        />
        <button
          className="flex items-center gap-2 bg-primary text-white p-2 rounded"
          type="submit"
        >
          <FaSearch size={14} />
          <span className="hidden sm:inline xl:hidden 2xl:inline">
            {t("search")}
          </span>
        </button>
      </form>
    </>
  );
};

export default Search;
